#!/bin/bash
############################################################################
# installARemb.sh - A menu driven shell script to install ARemb on an APC. #
# Author: Peter Maier                                                      #
# Date: 12/Jun/2016                                                        #
############################################################################

# print debug messages and optionally just simulate the action
DEBUG=
NO_ACTION=
NO_ACTION_PARAM="NO_ACTION"

while [ -n "$1" ]
do
	if [ "$1" = $NO_ACTION_PARAM ]; then
		NO_ACTION=$1
	fi
	shift
done

SETUP_DIR="/Setup"
ROOT_DIR="/root"
BASEAR_DIR=$ROOT_DIR"/BaseAR"
SHELL_SCRIPT_DIR=$ROOT_DIR"/scripts"
BOOT_INFO_DIR=$SHELL_SCRIPT_DIR

BOOT_INFO_RESULT_FILE=$BOOT_INFO_DIR/RESULTS.txt
INSTALL_LOG_FILE=$SETUP_DIR"/install_log.txt"

# Define variables
LSB=/usr/bin/lsb_release

# include scripts
source "${SHELL_SCRIPT_DIR}/helper.sh"
source "${SHELL_SCRIPT_DIR}/handlePartition.sh"
source "${SHELL_SCRIPT_DIR}/installInitialARemb.sh"

AREMB_DIALOG_INSTALL_TITLE="B&R ARemb Installation"
AREMB_DIALOG_INSTALL_BACKTITLE="B&R ARemb Installation"
AREMB_DIALOG_PARTITIONS_TITLE="Show partitions"
AREMB_DIALOG_EXISTING_EXTENDED_PARTITION_TITLE="Already existing extended partition"
GO_BACK_LABEL="Go back"
OK_LABEL="OK"
INSTALL_LABEL="Install"
EXIT_LABEL="Exit"
NEXT_LABEL="Next"
RESTART_LABEL="Restart"

DIALOG_INSTALL_STATUS=0				#$DIALOG_OK
DIALOG_OK_STATUS=0					#$DIALOG_OK
DIALOG_BACK_STATUS=0				#$DIALOG_OK
DIALOG_NEXT_STATUS=3				#$DIALOG_EXTRA
DIALOG_CANCEL_STATUS=1				#$DIALOG_CANCEL

# constants
MAIN_MENU_IDX=1
INST_CFG_AREMB_IDX=3

INST_CFG_SUMMARY_IDX=5
INST_START_IDX=6
INST_FINISHED_IDX=7
INST_FAILED_IDX=8

SHOW_PARTITIONS_IDX=40

pageIdx=$MAIN_MENU_IDX

function _readPartitionLabel() {
	local __label=$1
	local device=$2
	local partNr=$3
	local fileSystem=$4

	logWriteDebugFunction "_readPartitionLabel($__label, $device, $partNr, $fileSystem)"
	
	checkLastParam $partNr "no partition number given."

	local label=
	local arembSetup="0"
	if [[ "$fileSystem" =~ "fat" ]]; then		#fat16 and fat32
		label=$(dosfslabel $device$partNr)

	elif [ "$fileSystem" = "ntfs" ]; then
		label=$(ntfslabel $device$partNr)

	elif [ "$fileSystem" = "ext2" ] || [ "$fileSystem" = "ext3" ] || [ "$fileSystem" = "ext4" ]; then
		label=$(e2label $device$partNr)
	fi

	eval "${__label}='${label}'"
}

function _getFreeSpace()
{
	local __freeSpace=$1
	local devicePartNr=$2
	local free=
	local freeDir
	local i

	checkLastParam $devicePartNr "no device partNr given."

	mountPartition freeDir $devicePartNr

	local OIFS=$IFS
	IFS=$'\n'
	local dfOutput=$(df -m $freeDir)
	local arr=$dfOutput
	for i in $arr
	do
		if [ -n "$i" ]; then
			if [ "${i:0:10}" = "Filesystem" ]; then
				availableStart=$(echo "$i" | awk '{ print index ($0, "Available")-1;}')
			else
				free=$(echo ${i:$availableStart:9} | sed 's/ //g')
				break
			fi
		fi
	done
	IFS=$OIFS
	umount $freeDir
	eval "${__freeSpace}='${free}'"
}

function _readSingleDrive() {
	local drive=$1
	local driveSize=$2
	local driveLogSectorSize=$3
	local drivePhysSectorSize=$4
	local drivePartTable=$5
	local driveDesc=$6
	local driveNumber=$7
	local driveMBR=$8
	local i

	logWriteDebugFunction "_readSingleDrive ($drive, $driveSize, $driveLogSectorSize, \
		$drivePhysSectorSize, $drivePartTable, $driveDesc, $driveNumber, $driveMBR)"
		
	checkLastParam $driveMBR "no drive MBR given."

	# read partition info
	local OIFS=$IFS
	IFS=$'\n'
	local partedOutput=$(parted -s $drive unit MB print free 2> /dev/null)
	local arr=$partedOutput
	local nextIsAPartition="0"
	local extendedPartNr=
	for i in $arr
	do
		if [ -n "$i" ]; then
			firstHead=$(echo $i | cut -d ' ' -f 1)
			if [ "$firstHead" = "Number" ]; then
				nextIsAPartition="1"
				extendedPartNr=
				ignoreDisk=
				local numberStart=$(echo "$i" | awk '{ print index ($0, "Number");}')		#index of "Number"
				local startStart=$(echo "$i" | awk '{ print index ($0, "Start")-1;}')
				local endStart=$(echo "$i" | awk '{ print index ($0, "End")-1;}')
				local sizeStart=$(echo "$i" | awk '{ print index ($0, "Size")-1;}')
				local typeStart=$(echo "$i" | awk '{ print index ($0, "Type")-1;}')
				local fileSystemStart=$(echo "$i" | awk '{ print index ($0, "File system")-1;}')
				local flagsStart=$(echo "$i" | awk '{ print index ($0, "Flags")-1;}')
				logWriteDebug "number:$numberStart, start:$startStart, end:$endStart, size:$sizeStart, type:$typeStart, fileSystem:$fileSystemStart, flags:$flagsStart"
			else
				if [ "$nextIsAPartition" = "1" ]; then
					local partNr=$(echo ${i:$numberStart} | cut -d ' ' -f 1)
					if [ -n "$partNr" ]; then
						getPartitionPrefix partitionPrefix $drive
						partNr=$partitionPrefix$partNr
					fi
					local start=$(echo ${i:$startStart} | cut -d ' ' -f 1)
					local end=$(echo ${i:$endStart} | cut -d ' ' -f 1)
					local size=$(echo ${i:$sizeStart} | cut -d ' ' -f 1)
					local type=$(echo ${i:$typeStart} | cut -d ' ' -f 1)
					local fileSystem=$(echo ${i:$fileSystemStart} | cut -d ' ' -f 1)
					local flags=$(echo ${i:$flagsStart} | cut -d ' ' -f 1)

					# hide partitions < 2MB
					local sizeInMB=${size//MB/}	# "MB" => ""
					if (( ( $(echo "${sizeInMB} < 2" | bc -l) ) )); then
						continue
					fi

					local correspondingExtPartNr=$extendedPartNr
					local partLabel=
					local freeSpaceInMb=

					if $( echo $fileSystem | grep -q 'linux-swap' ); then
						partLabel="swap"
					elif [ "$type" = "extended" ]; then
						extendedPartNr=${#partitionDevice[@]}
						partLabel="extended"
					elif [ -n "$partNr" ]; then
						_readPartitionLabel partLabel $drive $partNr $fileSystem
						_getFreeSpace freeSpaceInMb $drive$partNr
					else
						partLabel=
						freeSpaceInMb=$sizeInMB
					fi

					if [ -z "$freeSpaceInMb" ]; then
						freeSpaceInMb="0"
					fi

					if [ -z "$partLabel" ]; then
						partLabel="untitled"
					fi

					# hide ARemb Install Stick and B&R dongle
					if [ "${partLabel}" == "AREMB-SETUP" ] || [ "${partLabel}" == "CODEMETER  " ]; then
						ignoreDisk=1
					fi

					if [ "$ignoreDisk" = "1" ]; then
						continue
					fi

					local shortPartName=${drive:5}$partNr
					# search for line containing e.g. "sda1: ___", after that the first occurrence of "Boot files" 
					
					# save parameters in global arrays
					partitionDevice=("${partitionDevice[@]}" "$drive")
					partitionDiskNumber=("${partitionDiskNumber[@]}" "$driveNumber")
					partitionDeviceSize=("${partitionDeviceSize[@]}" "$driveSize")
					partitionDeviceLogSectorSize=("${partitionDeviceLogSectorSize[@]}" "$driveLogSectorSize")
					partitionDevicePhysSectorSize=("${partitionDevicePhysSectorSize[@]}" "$drivePhysSectorSize")
					partitionDevicePartTable=("${partitionDevicePartTable[@]}" "$drivePartTable")
					partitionDeviceDesc=("${partitionDeviceDesc[@]}" "$driveDesc")

					partitionPartNr=("${partitionPartNr[@]}" "$partNr")
					partitionFs=("${partitionFs[@]}" "$fileSystem")
					partitionType=("${partitionType[@]}" "$type")
					local sizeInMb=${size//MB/}	# "MB" => ""
					partitionSize=("${partitionSize[@]}" "$sizeInMb")
					partitionFree=("${partitionFree[@]}" "$freeSpaceInMb")
					partitionStart=("${partitionStart[@]}" "$start")
					partitionEnd=("${partitionEnd[@]}" "$end")
					partitionLabel=("${partitionLabel[@]}" "$partLabel")
					partitionFlag=("${partitionFlag[@]}" "$flags")
					partitionExtPartNr=("${partitionExtPartNr[@]}" "$correspondingExtPartNr")

					partitionDeviceMBR=("${partitionDeviceMBR[@]}" "$driveMBR")

					if [ "1" = "0" ]; then
						echo "drive:$drive"
						echo "diskNumber:$driveNumber"
						echo "driveSize:$driveSize"
						echo "driveLogSectorSize:$driveLogSectorSize"
						echo "drivePhysSectorSize:$drivePhysSectorSize"
						echo "drivePartTable:$drivePartTable"
						echo "driveDesc:$driveDesc"
						echo "partNr:$partNr"
						echo "fileSystem:$fileSystem"
						echo "type:$type"
						echo "sizeInMb:$sizeInMb (MB)"
						echo "freeSpaceInMb:$freeSpaceInMb (MB)"
						echo "start:$start"
						echo "end:$end"
						echo "partLabel:$partLabel"
						echo "flags:$flags"
						echo "correspondingExtPartNr:$correspondingExtPartNr"
						echo "driveMBR:$driveMBR"
						pause
					fi
				fi
			fi
		fi
	done
	
	# komplett leere CFast ohne part table ohne partitions
	if [ "$nextIsAPartition" = "0" ]; then
		partitionDevice=("${partitionDevice[@]}" "$drive")
		partitionDiskNumber=("${partitionDiskNumber[@]}" "$driveNumber")
		partitionDeviceSize=("${partitionDeviceSize[@]}" "$driveSize")
		partitionDeviceLogSectorSize=("${partitionDeviceLogSectorSize[@]}" "$driveLogSectorSize")
		partitionDevicePhysSectorSize=("${partitionDevicePhysSectorSize[@]}" "$drivePhysSectorSize")
		partitionDevicePartTable=("${partitionDevicePartTable[@]}" "$drivePartTable")
		partitionDeviceDesc=("${partitionDeviceDesc[@]}" "$driveDesc")

		partitionPartNr=("${partitionPartNr[@]}" "1")
		partitionFs=("${partitionFs[@]}" "-")
		partitionType=("${partitionType[@]}" "-")
		partitionSize=("${partitionSize[@]}" "$driveSize")
		partitionFree=("${partitionFree[@]}" "$driveSize")
		partitionStart=("${partitionStart[@]}" "0")
		partitionEnd=("${partitionEnd[@]}" "0")
		partitionLabel=("${partitionLabel[@]}" "-")
		partitionFlag=("${partitionFlag[@]}" "-")
		partitionExtPartNr=("${partitionExtPartNr[@]}" "-")

		partitionDeviceMBR=("${partitionDeviceMBR[@]}" "-")
	fi
	
	IFS=$OIFS
}

function _readDrives(){
	# delete arrays contents
	partitionDevice=()
	partitionDiskNumber=()
	partitionDeviceSize=()
	partitionDeviceLogSectorSize=()
	partitionDevicePhysSectorSize=()
	partitionDevicePartTable=()
	partitionDeviceDesc=()
	partitionDeviceMBR=()

	partitionPartNr=()
	partitionFs=()
	partitionType=()
	partitionSize=()
	partitionFree=()
	partitionStart=()
	partitionEnd=()
	partitionLabel=()
	partitionFlag=()
	partitionExtPartNr=()

	local drives=()
	local driveNumbers=()
	local driveSizes=()
	local driveLogSectorSizes=()
	local drivePhysSectorSizes=()
	local drivePartTables=()
	local driveDescs=()
	local driveMBRs=()
	local i=

	local usbDevice=$(mount | grep "/Setup" | cut -d ' ' -f 1)
	usbDevice=${usbDevice:0:8}		#removing partition number

	# read all drives
	local nextIsADevice="0"
	local OIFS=$IFS
	IFS=$'\n'

	local partedOutput=$(parted -l -m -s 2> /dev/null)
	local arr=$partedOutput
	for i in $arr
	do
		if [ -n "$i" ]; then
			if [ "$i" = "BYT;" ]; then
				nextIsADevice="1"
				let cnt=cnt+1
			else
				if [ "$nextIsADevice" = "1" ]; then
					nextIsADevice="0"
					local device=$(echo $i | cut -d ':' -f 1)
					local size=$(echo $i | cut -d ':' -f 2)
					local logSectorSize=$(echo $i | cut -d ':' -f 4)
					local physSectorSize=$(echo $i | cut -d ':' -f 5)
					local partTable=$(echo $i | cut -d ':' -f 6)
					local desc=$(echo $i | cut -d ':' -f 7)
					desc=${desc//;/}
					
					# filter out usb install stick device
					if [ "$usbDevice" != "$device" ] ; then
						_getDiskNumberOfDevice deviceNumber $device

						drives=("${drives[@]}" "$device")
						driveNumbers=("${driveNumbers[@]}" "$deviceNumber")
						driveSizes=("${driveSizes[@]}" "$size")
						driveLogSectorSizes=("${driveLogSectorSizes[@]}" "$logSectorSize")
						drivePhysSectorSizes=("${drivePhysSectorSizes[@]}" "$physSectorSize")
						drivePartTables=("${drivePartTables[@]}" "$partTable")
						driveDescs=("${driveDescs[@]}" "$desc")
					fi
				fi
			fi
		fi
	done
	IFS=$OIFS

	# read boot info (MBR, grub.cfg)
	rm -f $BOOT_INFO_RESULT_FILE
	$SHELL_SCRIPT_DIR/bootinfoscript

	# read partition info of each drive
	for (( i=0; i<${#drives[@]}; i++ )) 
	do		
		driveMBR=$(grep "=>" $BOOT_INFO_RESULT_FILE | grep ${drives[$i]} | cut -d ' ' -f 3)
		if [ -z "$driveMBR" ]; then
			driveMBR="-"
		fi
		driveMBRs=("${driveMBRs[@]}" "$driveMBR")

		_readSingleDrive "${drives[$i]}" "${driveSizes[$i]}" "${driveLogSectorSizes[$i]}" \
		"${drivePhysSectorSizes[$i]}" "${drivePartTables[$i]}" "${driveDescs[$i]}" "${driveNumbers[$i]}" "${driveMBRs[$i]}"
	done

	if [ -n "$DEBUG_MODE" ]; then
		echo "device information:"
		for (( i=0; i<${#drives[@]}; i++ )) 
		do
			echo "${drives[$i]} ${driveSizes[$i]} ${driveLogSectorSizes[$i]} ${drivePhysSectorSizes[$i]} ${drivePartTables[$i]} ${driveDescs[$i]} ${driveNumbers[$i]} ${driveMBRs[$i]}"
		done
	fi

	### for debugging
	if [ -n "$DEBUG_MODE" ]; then
		echo "partition information:"
		# read partition info of each drive
		for (( i=0; i<${#partitionDevice[@]}; i++ )) 
		do
			echo "${partitionDevice[$i]} ${partitionDiskNumber[$i]} ${partitionDeviceSize[$i]} ${partitionDeviceLogSectorSize[$i]} ${partitionDevicePhysSectorSize[$i]} ${partitionDevicePartTable[$i]} ${partitionDeviceDesc[$i]} ${partitionDeviceMBR[$i]} ${partitionDeviceGrubPartNr[$i]} ${partitionDeviceGrubCfg[$i]} ${partitionDeviceGposString[$i]}"
			echo "    ${partitionPartNr[$i]} ${partitionFs[$i]} ${partitionType[$i]} ${partitionSize[$i]} ${partitionFree[$i]} ${partitionStart[$i]} ${partitionEnd[$i]} ${partitionLabel[$i]} ${partitionFlag[$i]} ${partitionExtPartNr[$i]}"
		done
		pause
	fi
}

function _printPartitions(){
	if [ -n "$DEBUG_MODE" ]; then
		echo "name: $1, cnt: $2, default: $3"
		shift
		shift
		shift

		while [ -n "$1" ]
		do
			echo "$1 $2"
			shift
			shift
		done
		echo ""
		if [ -n "$STEPPING_MODE" ]; then
			pause
		fi
	fi
}

function _getPartitionString() {
	local __partString=$1
	local i=$2
	
	local partStr=
	#deviceStringBrackets=" (drive:"${partitionDeviceSize[$i]}","${partitionDevicePartTable[$i]}","${partitionDeviceDesc[$i]}")"
	#deviceMore=${partitionFs[$i]}",start:"${partitionStart[$i]}","${partitionLabel[$i]}
	#deviceString=${partitionDevice[$i]}${partitionPartNr[$i]}","${partitionSize[$i]}

	#completeString=$deviceString","$deviceMore""$deviceStringBrackets
	#addPartString="[${partitionDeviceMBR[$i]},${partitionDeviceGrubCfg[$i]}]"

	partStr="${partitionDeviceDesc[$i]} (${partitionDeviceSize[$i]})"
	if [ -z "${partitionType[$i]}" ]; then
		partStr="${partStr}, unallocated"
	elif [ "${partitionType[$i]}" = "extended" ]; then
		partStr="${partStr}, extended partition"
	else
		partStr="${partStr}, ${partitionLabel[$i]}"
	fi

	#if (( (${partitionPartNr[$i]} >= 5) )); then			# logical drives
		#	partStr="  "${partitionLabel[$i]}
	#fi

	partStr="disk${partitionDiskNumber[$i]}: $partStr, ${partitionSize[$i]}MB (free:${partitionFree[$i]}MB) ${partitionFs[$i]}"
	eval "${__partString}='${partStr}'"
}

function _getARembInstallationCandidate() {
	local __driveString=$1
	local driveNr=$2
	
	local j=
	local driveString=
	
	for (( j=0; j<${#partitionDevice[@]}; j++ ))
	do
		# loop to drive number
		if [ "${partitionDiskNumber[$j]}" = "$driveNr" ]; then
			driveString="disk$driveNr: ${partitionDeviceDesc[$j]} (${partitionDeviceSize[$j]}, ${partitionFree[$j]}MB free) "
			break
		fi
	done
	eval "${__driveString}='${driveString}'"
}

# all disk candidates on which ARemb can be installed
# i.e.: there is no extended partition on the disk 
function _createARembInstallationCandidates(){
	AREMB_CAND_PART_COUNTER=1
	AREMB_CAND_PART_LIST=()  # variable for possible bootable partitions
	AREMB_CAND_PART_IDX_LIST=()
	defaultCandPartitionNr=0

	for (( i=0; i<${#partitionDevice[@]}; i++ ))
	do
		_getARembInstallationCandidate candDriveString $i

		if [ -n "$candDriveString" ]; then
			AREMB_CAND_PART_LIST=("${AREMB_CAND_PART_LIST[@]}" "$AREMB_CAND_PART_COUNTER")
			AREMB_CAND_PART_LIST=("${AREMB_CAND_PART_LIST[@]}" "$candDriveString")
			AREMB_CAND_PART_IDX_LIST=("${AREMB_CAND_PART_IDX_LIST[@]}" "$i")
			if [ "$defaultCandPartitionNr" = "0" ]; then
					let defaultCandPartitionNr=$AREMB_CAND_PART_COUNTER
			fi
			let AREMB_CAND_PART_COUNTER=AREMB_CAND_PART_COUNTER+1
		fi
	done
	_printPartitions "CAND_PART" $AREMB_CAND_PART_COUNTER $defaultCandPartitionNr ${AREMB_CAND_PART_IDX_LIST[@]} ${AREMB_CAND_PART_LIST[@]}
}

function _createPartitionInfoStrings(){
	ALL_PART_COUNTER=1
	ALL_PART_LIST=()  # variable where we will keep the list entries for menu dialog
	ALL_PART_IDX_LIST=()

	# all bootable partitions
	BOOT_PART_COUNTER=1
	BOOT_PART_LIST=()
	BOOT_PART_IDX_LIST=()
	defaultbootPartitionNr=0

	partitionComplete=()
	partitionCnt=0
	for (( i=0; i<${#partitionDevice[@]}; i++ ))
	do
		_getPartitionString candPartString $i
		
		# all partitions
		ALL_PART_LIST=("${ALL_PART_LIST[@]}" "$ALL_PART_COUNTER")
		ALL_PART_LIST=("${ALL_PART_LIST[@]}" "$candPartString")
		ALL_PART_IDX_LIST=("${ALL_PART_IDX_LIST[@]}" "$i")
		let ALL_PART_COUNTER=ALL_PART_COUNTER+1

		# all bootable partitions
		if [[ "${partitionFlag[$i]}" =~ "boot" ]]; then
			BOOT_PART_LIST=("${BOOT_PART_LIST[@]}" "$BOOT_PART_COUNTER")
			BOOT_PART_LIST=("${BOOT_PART_LIST[@]}" "$candPartString")
			BOOT_PART_IDX_LIST=("${BOOT_PART_IDX_LIST[@]}" "$i")
			if [ "$defaultbootPartitionNr" = "0" ]; then
				let defaultbootPartitionNr=$BOOT_PART_COUNTER
			fi
			let BOOT_PART_COUNTER=BOOT_PART_COUNTER+1
		fi

		partitionComplete=("${partitionComplete[@]}" "$candPartString")
		let partitionCnt=$partitionCnt+1
	done

	_printPartitions "ALL_PART" $ALL_PART_COUNTER 0 ${ALL_PART_IDX_LIST[@]} ${ALL_PART_LIST[@]}
	_printPartitions "BOOT" $BOOT_PART_COUNTER $defaultbootPartitionNr ${BOOT_PART_IDX_LIST[@]} ${BOOT_PART_LIST[@]}
	_printPartitions "ALL" $partitionCnt 0 ${partitionComplete[@]}
}

# ignore CTRL+C, CTRL+Z and quit singles using the trap
trap '' SIGINT SIGQUIT 

trap ctrl_z SIGTSTP 
function ctrl_z() { echo "** trapped Crtl-Z";   exit 0; }
 
trap ctrl_c INT
function ctrl_c() { echo "** Trapped CTRL-C";   exit 0; }

# for testing, on real system wait 5 seconds only
myhostname=$(hostname)
if [ "$myhostname" = "buildroot" ]; then
	: #sleep 5 TODO enablen
fi

tempfile=./tempfile.txt

#############################################
# Receives the result given in the file in the first parameter from the dialog
function _receiveResult(){
   retv=$?
   choice=$(cat $1)
   [ $retv -eq 1 -o $retv -eq 255 ] && exit
}

# Main menu dialog
function _showMainMenuDialog(){
	local menuList=()
	menuList=("1" "Install" "2" "Show partitions")
	
    dialog --title "$AREMB_DIALOG_INSTALL_TITLE" --backtitle "$AREMB_DIALOG_INSTALL_BACKTITLE" \
        --cancel-label "Exit" \
        --menu "\nPlease choose an option:\n \n" 15 55 3 "${menuList[@]}" 2> $tempfile
	_receiveResult $tempfile

	case $choice in
    	1) pageIdx=$INST_CFG_AREMB_IDX ;;
    	2) pageIdx=$SHOW_PARTITIONS_IDX ;;
	esac
}

# Shows all existing drives with all partitions
function _showPartitionInfoDialog() {
	title=$1
	backtitle=$2
	menu=$3
	defaultitem=$4
	count=$5
	shift
	shift
	shift
	shift
	shift

	dialog --title "$title" --backtitle "$backtitle" \
                --ok-label "$GO_BACK_LABEL" --extra-button --extra-label "$NEXT_LABEL" --default-button extra \
                --default-item "$defaultitem" \
                --menu "$menu" 20 120 $count \
"$@" 2> $tempfile
    _receiveResult $tempfile

	if (( ("$retv" == "$DIALOG_NEXT_STATUS") )); then
		pageIdx=$choice
	fi
}

function _showInstCfgARembPartDialog(){
    _showPartitionInfoDialog "$AREMB_DIALOG_INSTALL_TITLE - Select disk for ARemb" "$AREMB_DIALOG_INSTALL_BACKTITLE" "\
\nPlease select the disk for ARemb: \n \n"\
 "$defaultCandPartitionNr" $AREMB_CAND_PART_COUNTER "${AREMB_CAND_PART_LIST[@]}"

	if (( ("$retv" == "$DIALOG_NEXT_STATUS") )); then
		let choice=$choice-1
		ARembPartitionNr=${AREMB_CAND_PART_IDX_LIST[$choice]}
		pageIdx=$INST_CFG_SUMMARY_IDX			#ok
	else
		pageIdx=$INST_CFG_BOOT_IDX				#back
		pageBackButton=1
	fi
}

function _showPartitionsDialog(){
    _showPartitionInfoDialog "$AREMB_DIALOG_PARTITIONS_TITLE" "$AREMB_DIALOG_INSTALL_BACKTITLE" "\
\nAvailable drives: \n \n"\
 "0" $ALL_PART_COUNTER "${ALL_PART_LIST[@]}"

    pageIdx=$MAIN_MENU_IDX
}

function _showInstCfgSummaryDialog(){
	local arembDevice=${partitionDevice[$ARembPartitionNr]}
	local arembDisk=${partitionDiskNumber[$ARembPartitionNr]}
	local arembFree=${partitionFree[$ARembPartitionNr]}
	local arembDiskDesc=${partitionDeviceDesc[$ARembPartitionNr]}
	local dialogText=
	local extPartIdx=

	dialogText=\
"\nPending operations:\n\n\
1. Create partition for ARemb on disk${arembDisk}: ${arembDiskDesc} \n\
2. Install ARemb\n \n \n
This will delete the whole disk and install ARemb. Do you still want to continue?"

	dialog --title "$AREMB_DIALOG_INSTALL_TITLE - Configuration summary" --backtitle "$AREMB_DIALOG_INSTALL_BACKTITLE" --cr-wrap \
                --ok-label "$INSTALL_LABEL" \
                --extra-button --extra-label "$GO_BACK_LABEL" \
                --no-label "$EXIT_LABEL" \
                --default-button extra \
                --yesno \
                "$dialogText" 17 120
                _receiveResult $tempfile

    if (( ("$retv" == "$DIALOG_INSTALL_STATUS") )); then
    	pageIdx=$INST_START_IDX					#install
	else											#back
		pageBackButton=1
		pageIdx=$INST_CFG_AREMB_IDX
	fi
}

function _install(){
	local arembDevice=${partitionDevice[$ARembPartitionNr]}
	local arembPartNr=${partitionPartNr[$ARembPartitionNr]}

	deleteDevice $arembDevice

	createARembPartition $arembDevice
	
	resetMBR $arembDevice

	installInitialARemb $arembDevice

	if [ "$?" = "0" ]; then
		pageIdx=$INST_FINISHED_IDX
	else
		pageIdx=$INST_FAILED_IDX
	fi

	if [ -n "$DEBUG_MODE" ]; then
		pause
	fi
}

function _showErrorDialog(){
	local title=$1
	local text=$2

	checkLastParam $text "no text given."

    dialog --title "$title" --backtitle "$AREMB_DIALOG_INSTALL_BACKTITLE" --cr-wrap \
                --msgbox "\n$text\n\n\
Press Ok to continue." 10 80
                _receiveResult $tempfile
}

# main logic
#sleep 5			# waiting for 5 seconds

if [ -z "$DEBUG_MODE" ]; then
	dmesg -n 1
fi
loadkeys de
#clear

# read drives and partitions
partitionsChanged=1

logWriteStart

while true
do
	if [ "$partitionsChanged" = "1" ]; then
		partitionsChanged="0"
		_readDrives
		
		_createARembInstallationCandidates
		
		_createPartitionInfoStrings
	fi

	case $pageIdx in
    $MAIN_MENU_IDX)
        pageBackButton=0
        _showMainMenuDialog ;;

    $INST_CFG_AREMB_IDX)
        if [ "$AREMB_CAND_PART_COUNTER" = "2" ]; then			#index faengt bei 1 an
            if (( ("$pageBackButton" == 1) )); then
                pageIdx=$MAIN_MENU_IDX
            else
            	ARembPartitionNr=${AREMB_CAND_PART_IDX_LIST[0]}
                pageIdx=$INST_CFG_SUMMARY_IDX				#ueberspringen
            fi
        elif [ "$AREMB_CAND_PART_COUNTER" == "1" ];then			#liste leer => kein ARemb candidate vorhanden
			_showErrorDialog "$AREMB_DIALOG_INSTALL_TITLE" \
				"There is no disk available to install B&R Hypervisor."
			pageIdx=$MAIN_MENU_IDX
        else
            pageBackButton=0
            _showInstCfgARembPartDialog
        fi;;
    $INST_CFG_SUMMARY_IDX)           _showInstCfgSummaryDialog ;;
    $INST_START_IDX)                   partitionsChanged="1" && _install ;;
    $INST_FINISHED_IDX)              showRebootOkDialog \
                                        "$AREMB_DIALOG_INSTALL_TITLE finished"\
                                        "Installation successfully completed.";;
    $INST_FAILED_IDX)                showRebootErrorDialog \
                                        "$AREMB_DIALOG_INSTALL_TITLE failed"\
                                        "Installation failed.";;

    $SHOW_PARTITIONS_IDX)        _showPartitionsDialog ;;
    *)			echo "unhandled pageIdx: $pageIdx"; exit 1;
   esac
done
