#!/bin/bash
# installARemb.sh - A menu driven shell script to install ARemb on an APC.
# Author: Peter Maier
# Date: 12/Jun/2016

WINDOWS_TAG="Windows"
LINUX_TAG="Linux"
AR_SYSTEM_DIR="/SYSTEM"
GRUB_DIR=$AR_SYSTEM_DIR"/GRUB"
RTH_DIR=$AR_SYSTEM_DIR"/RTH"

LINUX_VMLINUZ_NAME="vmlinuz"
LINUX_INITRAMFS_NAME="initrd.img"

LOG_FUNC="func :"
LOG_DBG="log  :"
LOG_INFO="info :"
LOG_ERROR="error:"
LOG_FATAL="fatal:"
LOG_CALL="call :"

mountedDirIndex=1

# Purpose: Display pause prompt
# $1-> Message (optional)
function pause(){
	local message="$@"
	[ -z $message ] && message="Press [Enter] key to continue, [ESC] to exit..."
	read -p "$message" readEnterKey
	if [ "$readEnterKey" = $'\e' ]; then		# ESC
		exit 0;
	fi
}

function logWriteStackTrace()
{
	local cntBashSrc="2"
	local cntBashLineNo="1"

	logWriteDebug "error stack trace:"
	while [ "$cntBashSrc" -lt "${#BASH_SOURCE[@]}" ]
	do
		logWriteDebug "    ${BASH_SOURCE[$cntBashSrc]}:${BASH_LINENO[$cntBashLineNo]}"
		let cntBashSrc=$cntBashSrc+1
		let cntBashLineNo=$cntBashLineNo+1
	done
}

function getSubStringContaining (){
	local __result=$1
	local string=$2
	local searchString=$3
	local tmpString=
	local result=

	checkLastParam "$searchString" "no search string given. getSubStringContaining($__result, $string, $searchString)"

	string=$(echo $string | sed 's/  */ /g' | sed -e 's/^[ \t]*//')   #trim spaces, remove leading spaces
	while [ -n "$string" ]
	do 
		tmpString=$(echo $string | cut -d ' ' -f1)

		if [ -z "$tmpString" ]; then
			break;
		else
			if [[ "$tmpString" =~ "$searchString" ]]; then
				result=$tmpString
				break;
			fi
		fi 
		newString=$(echo $string | cut -d ' ' -f2-)		# "shift"
		if [ "$newString" = "$string" ]; then
			string=
		else
			string=$newString
		fi
	done

	eval "$__result='$result'"
}


# extracts the i-th substring with given delimiter 
# $1 ... i 
# $2 ... delim
# $3 ... string
# return ... value in RESULTING_SUBSTRING
function getIthSubstringWithDel(){
	local idx=0
	local cnt=0
	local tmpString=$3
	while [ "$cnt" -lt "$1" ]
	do 
		idx=$(echo `expr index "$tmpString" $2`)		# index of ":"
		tmpString=${tmpString:$idx}
		let cnt=cnt+1
	done

	idx=$(echo `expr index "$tmpString" $2`)		# index of ":"
	let idx=idx-1
	tmpString=${tmpString:0:$idx}
	RESULTING_SUBSTRING=$tmpString
}
	
function isdigit_within(){    # Tests whether *entire string* is numerical and within the borders.
             # In other words, tests for integer variable.
	SUCCESS=0
	FAILURE=1

	digit=$1
	min=$2
	max=$3

	[ $# -eq 3 ] || return $FAILURE

	case $digit in
    	*[!0-9]*|"") return $FAILURE;;
            	*) :;;
	esac

	if (( ("$min" <= "$digit") && ("$digit" <= "$max") )); then
		return $SUCCESS
	else
		return $FAILURE
	fi 
}

function mountPartition() {
	local __mountedDir=$1
	local devicePartNr=$2

	local mountedDir="/mnt/part"

	checkLastParam "$devicePartNr" "no device and partition number given. mountPartition ($__mountedDir, $devicePartNr)"

	mountedDir=$mountedDir$mountedDirIndex
	let mountedDirIndex=$mountedDirIndex+1

	logWriteDebugFunctionCall "mkdir -p $mountedDir"
	logWriteDebugFunctionCall "mount $devicePartNr $mountedDir"
	local retv=$?
	if [ "$retv" = "0" ]; then
		local isNtfs=$(mount | grep $devicePartNr | grep "type ntfs")
		if [ -n "$isNtfs" ]; then		#remount
			logWriteDebugFunctionCall "umount $mountedDir"
			logWriteDebugFunctionCall "ntfsfix $devicePartNr"		#fix NTFS filesystem
			logWriteDebugFunctionCall "mount -t ntfs-3g $devicePartNr $mountedDir"
			retv=$?
		fi
	else
		logWriteDebugFunctionCall "ntfsfix $devicePartNr"		#fix NTFS filesystem
		logWriteDebugFunctionCall "mount -t ntfs-3g $devicePartNr $mountedDir"
		retv=$?
	fi

	eval $__mountedDir=$mountedDir
	return $retv
}

function umountDir() {
	local dir=$1

	checkLastParam "$dir" "no dir given. umountDir ($dir)"

	logWriteDebugFunctionCall "umount $dir"
	logWriteDebugFunctionCall "rmdir $dir"
}

if [ -n "$NO_ACTION" ]; then 
	noAction="on"
else
	noAction="off"
fi

function _getDiskNumberOfDevice(){
	local __num=$1
	local dev=$2

	logWriteDebugFunction "_getDiskNumberOfDevice($__num, $dev)"

	checkLastParam $dev "no device given."

	local letter=${dev:7}
	local num="0"
	if [ "$letter" = "a" ]; then
		num="0"
	elif [ "$letter" = "b" ]; then
		num="1"
	elif [ "$letter" = "c" ]; then
		num="2"
	elif [ "$letter" = "d" ]; then
		num="3"
	fi
	eval "${__num}='${num}'"	
}

stack=()
function pop() {
	local __stackValue=$1
	local value=
	local i=${#stack[@]}
	let i=$i-1
	
	value=${stack[$i]}
	unset 'stack[i]'
	
	eval $__stackValue=$value
	return 0
}

function push() {
	stack=("${stack[@]}" "$1")
}

function getPartitionPrefix()
{
	local __partPrefix=$1
	local device=$2
	local prefix=
	checkLastParam "$device" "no device given."
	
	if $( echo $device | grep -q '/dev/nvme' ); then
		prefix="p"
	fi
	eval $__partPrefix=$prefix
	return 0
}

function _showRebootDialog(){
    local title=$1
    local text=$2
    local isOkDialog=$3
    local doSilentReboot=$4

    checkLastParam $isOkDialog "ok dialog flag not given."

    if [ -n "$SILENT_INSTALL" ] && [ -n "$doSilentReboot" ]; then
        logWriteDebug "$isOkDialog reboot dialog: $title - $text (silent)"
        fakeResult $DIALOG_OK_STATUS "0"
    else
        logWriteDebug "$isOkDialog reboot dialog: $title - $text"
        dialog --title "$title" --backtitle "$AREMB_DIALOG_INSTALL_BACKTITLE" --cr-wrap \
                --ok-label "$RESTART_LABEL"\
                --msgbox "\n$text\n\nSystem will be restarted." 14 80
                _receiveResult $tempfile
    fi

    umount $SETUP_DIR 2> /dev/null
    $SHELL_SCRIPT_DIR/mtcxSwReset
    reboot
}

function showRebootOkDialog(){
    checkLastParam $2 "text not given."
    _showRebootDialog "$1" "$2" "ok" "$3"
}

function showRebootErrorDialog(){
    checkLastParam $2 "text not given."
    _showRebootDialog "$1" "$2" "error" "$3"
}

function logWrite() 
{
	local severity=$1
	local currentDate=$(date)
	
	checkLastParam "$severity" "no severity given."
	
	shift
	echo "$currentDate $severity $@" >> $INSTALL_LOG_FILE
}

function logWriteInfo() 
{
	logWrite "${LOG_INFO}" "$@"
}

function logWriteDebug()
{
	logWrite "${LOG_DBG}" "$@"
}

function logWriteError() 
{
	checkLastParam "$2" "just one parameter given"
	logWrite "${LOG_ERROR}" "$@"
	showRebootErrorDialog "$@"
}

function logWriteFatal()
{
	logWrite "${LOG_FATAL}" "${BASH_SOURCE[1]}:${BASH_LINENO[0]}: $@"
	logWriteStackTrace
	logWrite "${LOG_FATAL}" "see $INSTALL_LOG_FILE for further details"
	exit 1
}

# calls a command and writes a log entry before calling
function logWriteDebugFunctionCall()
{
	if [ -n "$DEBUG" ]; then
		logWrite "${LOG_CALL}" "$@" "(${BASH_SOURCE[1]}:${BASH_LINENO[0]})"
	else
		logWrite "${LOG_CALL}" "$@"
	fi
	if [ -z "$NO_ACTION" ]; then
		eval "$@"	&>> $INSTALL_LOG_FILE		# execute the given command with arguments
	fi
}

function logWriteDebugFunction()
{
	# only for debugging
	if [ -n "$DEBUG" ]; then
		logWrite "${LOG_FUNC}" "$@"
	fi
}

function logWriteFile()
{
	logWriteDebug "$1:"
	OIFS=$IFS
	IFS=$'\n'
	fileOutput=$(cat $1)
	arr=$fileOutput
	local i=0
	for i in $arr
	do
		logWriteDebug "    $i"
	done
	IFS=$OIFS
}

function logWriteCommandOutput()
{
	logWriteDebug "$1:"
	shift
	OIFS=$IFS
	IFS=$'\n'
	fileOutput=$(eval $@)
	arr=$fileOutput
	local i=0
	for i in $arr
	do
		logWriteDebug "    $i"
	done
	IFS=$OIFS
}

function logWriteLine()
{
	echo "===========================================================================" >> $INSTALL_LOG_FILE
}

function logWriteStart()
{
	echo "" >> $INSTALL_LOG_FILE
	echo "" >> $INSTALL_LOG_FILE
	logWriteLine
	logWriteLine
	if [ "$SILENT_INSTALL" = "y" ] || [ "$SILENT_INSTALL" = "Y" ] || [ "$SILENT_INSTALL" = "1" ] ;then
		SILENT_INSTALL="1"
	else
		SILENT_INSTALL=
	fi

	if [ -n "$SILENT_INSTALL" ];then
		logWriteInfo "Script started (silent installation)"
	else
		logWriteInfo "Script started."
	fi
	logWriteLine
	logWriteDebug "AR version: $AR_VERSION"			# defined in info.sh
	logWriteDebug "system info:"

	local i=
	local oemString=
	for (( i=1; i<=32; i++ ))
	do
		oemString=$(dmidecode --oem-string $i 2> /dev/null)
		if [ -n "$oemString" ]; then
			logWriteDebug "OEM string $i: $oemString"
		fi
	done

	local dmiString="BIOS:"
	local dmi1=$(dmidecode -s bios-version)
	local dmi2=$(dmidecode -s bios-release-date)
	local dmi3=$(dmidecode -s bios-vendor)
	local dmi4=
	local dmi5=
	logWriteDebug "BIOS: $dmi1 $dmi2 $dmi3 $dmi4 $dmi5"

	dmi1=$(dmidecode -s system-product-name)
	dmi2=$(dmidecode -s system-version)
	dmi3=$(dmidecode -s system-serial-number)
	dmi4=$(dmidecode -s system-uuid)
	dmi5=$(dmidecode -s system-manufacturer)
	logWriteDebug "system: $dmi1 $dmi2 $dmi3 $dmi4 $dmi5"

	dmi1=$(dmidecode -s baseboard-product-name)
	dmi2=$(dmidecode -s baseboard-version)
	dmi3=$(dmidecode -s baseboard-serial-number)
	dmi4=$(dmidecode -s baseboard-asset-tag)
	dmi5=$(dmidecode -s baseboard-manufacturer)
	logWriteDebug "baseboard: $dmi1 $dmi2 $dmi3 $dmi4 $dmi5"

	dmiString="chassis:"
	dmi1=$(dmidecode -s chassis-version)
	dmi2=$(dmidecode -s chassis-serial-number)
	dmi3=$(dmidecode -s chassis-type)
	dmi4=$(dmidecode -s chassis-asset-tag)
	dmi5=$(dmidecode -s chassis-manufacturer)
	logWriteDebug "chassis: $dmi1 $dmi2 $dmi3 $dmi4 $dmi5"

	dmiString="processor:"
	dmi1=$(dmidecode -s processor-version)
	dmi2=$(dmidecode -s processor-family)
	dmi3=$(dmidecode -s processor-frequency)
	dmi4=$(dmidecode -s processor-manufacturer)
	logWriteDebug "processor: $dmi1 $dmi2 $dmi3 $dmi4"

	logWriteCommandOutput "ethernet" "ifconfig"
	logWriteCommandOutput "drives" "parted -l"
}

function logWriteDeviceInfo()
{
	local dev=$1
	logWriteCommandOutput "information of device $dev updated" "parted -s $dev unit s print free"
}

function logWriteInstallStart()
{
	echo "" >> $INSTALL_LOG_FILE
	logWriteLine
	logWrite "${LOG_INFO}" "B&R Hypervisor installation started."
}

function logWriteInstallFinished()
{
	logWrite "${LOG_INFO}" "B&R Hypervisor installation finished."
	logWriteLine
}

function logWriteUninstallStart()
{
	echo "" >> $INSTALL_LOG_FILE
	logWriteLine
	logWrite "${LOG_INFO}" "B&R Hypervisor uninstallation started."
}

function logWriteUninstallFinished()
{
	logWrite "${LOG_INFO}" "B&R Hypervisor uninstallation finished."
	logWriteLine
}

function checkLastParam()
{
	if [ -z "$1" ]; then
		logWriteFatal "$2"
	fi

	if [ -z "$2" ]; then		# special case: just one parameter was given => print this one
		logWriteFatal "$1"
	fi
}
